from datetime import datetime, date
from passlib.context import CryptContext

# ---- Hash de contraseñas (bcrypt) ----
pwd_context = CryptContext(schemes=["bcrypt"], deprecated="auto")

def hash_password(password: str) -> str:
    # bcrypt admite hasta 72 bytes; evita contraseñas larguísimas
    pw = (password or "").strip()
    if len(pw.encode("utf-8")) > 72:
        pw = pw.encode("utf-8")[:72].decode("utf-8", errors="ignore")
    return pwd_context.hash(pw)

def verify_password(plain: str, hashed: str) -> bool:
    try:
        return pwd_context.verify((plain or "").strip(), hashed or "")
    except Exception:
        return False

# ---- Fechas: acepta DD/MM/YYYY o YYYY-MM-DD ----
def parse_date_any(s: str | None) -> date | None:
    if not s:
        return None
    s = s.strip()
    for fmt in ("%d/%m/%Y", "%Y-%m-%d"):
        try:
            return datetime.strptime(s, fmt).date()
        except ValueError:
            continue
    return None

# ---- Normalizar número para wa.me ----
def _digits_only(s: str) -> str:
    return "".join(ch for ch in s if ch.isdigit())

def normalize_peru_phone(s: str | None) -> str:
    """
    Devuelve solo dígitos para wa.me.
    - Si empieza con 51 -> lo deja.
    - Si tiene 9 dígitos y empieza con 9 -> antepone 51.
    - En otros casos, devuelve solo dígitos tal cual.
    """
    if not s:
        return ""
    d = _digits_only(s)
    if d.startswith("51"):
        return d
    if len(d) == 9 and d.startswith("9"):
        return "51" + d
    return d