from fastapi import FastAPI, Request, Form, Depends, HTTPException
from fastapi.responses import HTMLResponse, RedirectResponse
from fastapi.staticfiles import StaticFiles
from fastapi.templating import Jinja2Templates
from starlette.middleware.sessions import SessionMiddleware
from sqlalchemy.orm import Session
from sqlalchemy import func
from urllib.parse import quote
from datetime import date

from database import Base, engine, get_db
from models import Admin, Subscription
from utils import (
    hash_password, verify_password, parse_date_any, normalize_peru_phone
)

# ------------ App & middleware ------------
app = FastAPI(title="Tuki Subs")
app.add_middleware(SessionMiddleware, secret_key="tuki-secret-key-please-change")

# Archivos estáticos y plantillas
app.mount("/static", StaticFiles(directory="static"), name="static")
templates = Jinja2Templates(directory="templates")

# Crear tablas
Base.metadata.create_all(bind=engine)

# ------------ Helpers ------------
def get_current_admin(request: Request, db: Session) -> Admin | None:
    admin_id = request.session.get("admin_id")
    if not admin_id:
        return None
    return db.query(Admin).get(admin_id)

def require_admin(request: Request, db: Session) -> Admin:
    admin = get_current_admin(request, db)
    if not admin:
        raise HTTPException(status_code=303, detail="login", headers={"Location": "/login"})
    return admin

# Crear un admin por defecto si no hay
@app.on_event("startup")
def seed_admin():
    with next(get_db()) as db:
        count_admins = db.query(func.count()).select_from(Admin).scalar()
        if count_admins == 0:
            a = Admin(
                email="support@tuki.loonixx.com",
                password_hash=hash_password("Admin123")
            )
            db.add(a)
            db.commit()

# ------------ Rutas básicas ------------
@app.get("/", response_class=HTMLResponse)
def home():
    return RedirectResponse(url="/login")

@app.get("/login", response_class=HTMLResponse)
def login_get(request: Request):
    return templates.TemplateResponse("login.html", {"request": request})

@app.post("/login")
def login_post(
    request: Request,
    email: str = Form(...),
    password: str = Form(...),
    db: Session = Depends(get_db),
):
    admin = db.query(Admin).filter(Admin.email == email).first()
    if not admin or not verify_password(password, admin.password_hash):
        return templates.TemplateResponse(
            "login.html", {"request": request, "error": "Credenciales incorrectas"}
        )
    request.session["admin_id"] = admin.id
    return RedirectResponse(url="/admin", status_code=303)

@app.get("/logout")
def logout(request: Request):
    request.session.clear()
    return RedirectResponse(url="/login", status_code=303)

@app.get("/dashboard", response_class=HTMLResponse)
def dashboard(request: Request, db: Session = Depends(get_db)):
    admin = get_current_admin(request, db)
    return templates.TemplateResponse("dashboard.html", {"request": request, "user": admin})

# ------------ ADMIN: LISTA ------------
@app.get("/admin", response_class=HTMLResponse)
def subs_list(request: Request, db: Session = Depends(get_db), q: str = ""):
    require_admin(request, db)

    subs_query = db.query(Subscription)
    if q:
        subs_query = subs_query.filter(Subscription.phone.contains(q))
    subs = subs_query.all()

    items = []
    hoy = date.today()

    for s in subs:
        phone = normalize_peru_phone(s.phone or "")
        dias_restantes = None
        estado = "Sin fecha"
        color = "secondary"

        if s.expiry_date:
            dias_restantes = (s.expiry_date - hoy).days
            if dias_restantes > 2:
                estado, color = "Vigente", "success"
            elif 0 <= dias_restantes <= 2:
                estado, color = "Por vencer", "warning"
            else:
                estado, color = "Vencido", "danger"

        # --- Mensaje especial según estado ---
        if estado == "Vigente":
            msg = (
                f"✅ *{s.account_type.upper()} ACTIVO*\n"
                f"━━━━━━━━━━━━━━━━━━━━━━━\n"
                f"📅 *Vence el:* {s.expiry_date}\n"
                f"👤 *Perfil:* {s.profile}\n"
                f"🔐 *PIN:* {s.pin or '-'}\n"
                f"━━━━━━━━━━━━━━━━━━━━━━━\n"
                f"📱 Enviado desde *Tuki Subs*"
            )

        elif estado == "Por vencer":
            msg = (
                f"⚠️ *AVISO IMPORTANTE – TUKI SUBS*\n"
                f"━━━━━━━━━━━━━━━━━━━━━━━\n"
                f"Tu suscripción *{s.account_type.upper()}* vence el *{s.expiry_date}*.\n"
                f"Por favor renueva para mantener tu acceso sin interrupciones. 💻\n"
                f"━━━━━━━━━━━━━━━━━━━━━━━\n"
                f"💬 *Confirma tu renovación aquí.*"
            )

        elif estado == "Vencido":
            msg = (
                f"❌ *TU SUSCRIPCIÓN HA VENCIDO*\n"
                f"━━━━━━━━━━━━━━━━━━━━━━━\n"
                f"📺 *Cuenta:* {s.account_type.upper()}\n"
                f"📅 *Venció el:* {s.expiry_date}\n"
                f"━━━━━━━━━━━━━━━━━━━━━━━\n"
                f"🔁 *Para reactivarla, responde con:*\n"
                f"👉 **Renovar {s.account_type.upper()}**"
            )

        else:
            msg = (
                f"ℹ️ *Info de suscripción {s.account_type}*\n"
                f"Vence: *{s.expiry_date or '-'}*\n"
                f"Perfil: *{s.profile}*\n"
                f"PIN: *{s.pin or '-'}*"
            )

        wa_url = (
            f"https://wa.me/{phone}?text={quote(msg)}"
            if phone else f"https://wa.me/?text={quote(msg)}"
        )

        items.append({
            "sub": s,
            "wa_url": wa_url,
            "estado": estado,
            "color": color,
            "dias_restantes": dias_restantes
        })

    return templates.TemplateResponse(
        "admin_list.html",
        {"request": request, "items": items, "query": q}
    )

# ------------ ADMIN: CREAR ------------
@app.get("/admin/subs/new", response_class=HTMLResponse)
def subs_new_get(request: Request):
    return templates.TemplateResponse(
        "admin_new.html",
        {"request": request, "account_types": [
            "TUKI-NET", "TUKI-PRIME", "TUKI-MAX", "TUKI-DISNEY", "TUKI-PARAMOUNT", "TUKI-CRUNCHYROLL"
        ]}
    )

@app.post("/admin/subs/new")
def subs_new_post(
    request: Request,
    account_type: str = Form("TUKI-NET"),
    email: str = Form(""),
    password: str = Form(""),
    profile: int = Form(1),
    pin: str = Form(""),
    expiry: str = Form(""),
    phone: str = Form(""),
    db: Session = Depends(get_db),
):
    require_admin(request, db)

    sub = Subscription(
        account_type=account_type,
        email=email or None,
        password_hash=hash_password(password) if password else None,
        pw_note=password or None,
        profile=int(profile),
        pin=pin or None,
        expiry_date=parse_date_any(expiry),
        phone=phone or None,
    )
    db.add(sub)
    db.commit()
    return RedirectResponse(url="/admin", status_code=303)

# ------------ ADMIN: EDITAR ------------
@app.get("/admin/subs/{sub_id}/edit", response_class=HTMLResponse)
def subs_edit_get(sub_id: int, request: Request, db: Session = Depends(get_db)):
    require_admin(request, db)
    sub = db.query(Subscription).get(sub_id)
    if not sub:
        raise HTTPException(404, "No encontrado")
    return templates.TemplateResponse(
        "edit.html",
        {"request": request, "u": sub, "account_types": [
            "TUKI-NET", "TUKI-PRIME", "TUKI-MAX", "TUKI-DISNEY", "TUKI-PARAMOUNT", "TUKI-CRUNCHYROLL"
        ]},
    )

@app.post("/admin/subs/{sub_id}/edit")
def subs_edit_post(
    sub_id: int,
    request: Request,
    account_type: str = Form("TUKI-NET"),
    email: str = Form(""),
    profile: int = Form(...),
    pin: str = Form(""),
    expiry: str = Form(""),
    phone: str = Form(""),
    pw_note: str = Form(""),
    db: Session = Depends(get_db),
):
    require_admin(request, db)
    sub = db.query(Subscription).get(sub_id)
    if not sub:
        raise HTTPException(404, "No encontrado")

    sub.account_type = account_type
    sub.email = email or None
    sub.profile = int(profile)
    sub.pin = pin or None
    sub.expiry_date = parse_date_any(expiry)
    sub.phone = phone or None
    sub.pw_note = pw_note or None

    db.commit()
    return RedirectResponse(url="/admin", status_code=303)
